/*****************************************************************************
*  LPC15xx RC5 decoder demo program for MicroCore100 rev A board
*
*  1. Use SCT2 timer to capture and decode RC5 messages.
*  2. Use USART0 to print received RC5 frames to PC terminal
*****************************************************************************/
#include "global.h"

#define HB             889                                 // half bit time = 1778/2 = 889  usec
#define MIN_HB         640                                 // minimum half bit time  = 640  usec
#define MAX_HB         1140                                // maximum half bit time  = 1140 usec
#define MIN_2HB        1340                                // minimum full bit time  = 1340 usec
#define MAX_2HB        2220                                // maximum full bit time  = 2220 usec

uint8_t  RC5_System;                                       // format: 1 F C s4 s3 s3 s1 s0
uint8_t  RC5_Command;                                      // format: 0 0 c5 c4 c3 c2 c1 c0
uint8_t  RC5_flag;

static int32_t   low_time;                                 // to store and save captured low time
static int32_t   high_time;                                // to store and save captured high time
static uint8_t   prev;                                     // to save the previous half bit value
static uint16_t  frame;                           			// rc5 frame (max 14 bits)


static void RC5_Shift_Bit(uint8_t val)
{
    if (frame & 0x2000)                                    // frame full ?
        frame = 0;                                         // yes, ERROR
    else
        frame = (frame << 1) | val;                        // shift frame bit
}

/************************************************************************
; RC5_Decode (we only take action at a rising edge)
;
;  prev        Low Time        High Time     Action      new prev
; -------------------------------------------------------------------
;   0             0               0          Shift 0         0
;   0             0               1          Shift 1         1
;   0             1               0          -ERROR-         *
;   0             1               1          Shift 1,0       0
;   1             0               0          Shift 1         1
;   1             0               1          -ERROR-         *
;   1             1               0          Shift 1,0       0
;   1             1               1          -ERROR-         *
;
***********************************************************************/
static void RC5_Decode(void)
{
  uint8_t action;

    action = prev << 2;

// Check High Time

    if ((high_time > MIN_2HB) && (high_time < MAX_2HB))
        action = action | 1;                               // high_time = long
    else if (!((high_time > MIN_HB) && (high_time < MAX_HB)))
    {
        frame = 0;                                         // RC5 ERROR
        return;
    }

// Check Low Time

    if ((low_time > MIN_2HB) && (low_time < MAX_2HB))
        action = action | 2;                               // low_time = long
    else if (!((low_time > MIN_HB) && (low_time < MAX_HB)))
    {
        frame = 0;                                         // RC5 ERROR
        return;
    }

    switch (action)
    {
      case 0:  RC5_Shift_Bit(0);                           // short low, short high, shift 0
               break;
      case 1:  RC5_Shift_Bit(1);                           // short low, long high, shift 1
               prev = 1;                                   // new half bit is true
               break;
      case 2:  frame = 0;                                  // long low, short high, ERROR
      case 3:  RC5_Shift_Bit(1);                           // long low, long high, shift 1,0
               RC5_Shift_Bit(0);
               break;
      case 4:  RC5_Shift_Bit(1);                           // short low, short high, shift 1
               break;
      case 5:  frame = 0;                                  // short low, long high, ERROR
               break;
      case 6:  RC5_Shift_Bit(1);                           // long low, short high, shift 1,0
               RC5_Shift_Bit(0);
               prev = 0;                                   // new half bit is false
               break;
      case 7:  frame = 0;                                  // long low, long high, ERROR
      default: break;                                      // invalid
    }
}

void SCT2_IRQHandler(void)
{
    if (Chip_SCT_GetEventFlag(LPC_SCT2) & SCT_EVE_0)       // Event 0 ? (timeout)
    {                                                      // to guarantee a 12 msec idle time after last RC5 pulse
        if (frame & 0x2000)                                // frame (14 bits) full ?
        {
            RC5_Command = frame & 0x3F;                    // OK ! Save command byte
            RC5_System  = frame >> 6;                      // save system byte
            RC5_flag = 1;                                  // set event to trigger application
        }
        frame = 0;
        Chip_SCT_SetEventFlag(LPC_SCT2, SCT_EVT_0);        // clear event 0 flag
    }
    else                                                   // Event 2 (CTIN_0 rising edge)
    {
        if (frame)                                         // first pulse ?
        {
            high_time = LPC_SCT2->CAP[1].L;                // no, capture high time
            low_time  = LPC_SCT2->CAP[2].L;                // and capture low time
            RC5_Decode();                                  // and decode bit
        }
        else
        {
            prev  = 1;                                     // yes, assume half bit is true
            frame = 0x0001;                                // start bit received
        }
        Chip_SCT_SetEventFlag(LPC_SCT2, SCT_EVT_2);		   // clear event 2 flag
    }
}

void RC5_Init(void)
{
	Chip_Clock_EnablePeriphClock(SYSCTL_CLOCK_SCT2 );                	// enable the SCT2 clock

	Chip_SCT_Config(LPC_SCT2, 	SCT_CONFIG_32BIT_COUNTER 	|			// unified timers,
								SCT_CONFIG_AUTOLIMIT_L   	);

	Chip_SCT_SetControl(LPC_SCT2 , (SystemCoreClock/1000000-1) << 5);	// set prescaler, SCT clock = 1 MHz


    LPC_SCT2->REGMODE_L      = (1 << 1) | (1 << 2);        				// register pair 1 and 2 are capture

    Chip_SCT_SetMatchCountL(LPC_SCT2, SCT_MATCH_0, 12000);				// match 0 @ 12000/1MHz = 12 msec (timeout)
    Chip_SCT_SetMatchReload(LPC_SCT2, SCT_MATCH_0, 12000);



    Chip_SCT_EventState(LPC_SCT2, SCT_EVENT_0, ENABLE_STATE0);			// event 0 only happens in state 0

    Chip_SCT_EventControl(LPC_SCT2, SCT_EVENT_0, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH0	|	// MATCHSEL[3:0]   = related to match 0
                                                                          SCT_COMBMODE_MATCH	|	// COMBMODE[13:12] = uses match condition only
                                                                          SCT_STATELD_1		    |	// STATELD [14]    = STATEV is loaded into state
                                                                          SCT_STATEEV_0		    ));	// STATEV  [15]    = new state is 0




    Chip_SCT_EventState(LPC_SCT2, SCT_EVENT_1, ENABLE_STATE0);			// event 1 only happens in state 0

    Chip_SCT_EventControl(LPC_SCT2, SCT_EVENT_1, (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH6	|	// IOSEL   [9:6]   = SCT_IN0
                                                                          SCT_IOCOND_FALL	    |	// IOCOND  [11:10] = falling edge
                                                                          SCT_COMBMODE_IO	    |	// COMBMODE[13:12] = uses IO condition only
                                                                          SCT_STATELD_1		    |	// STATELD [14]    = STATEV is loaded into state
                                                                          SCT_STATEEV_0		    ));	// STATEV[ 15]     = new state is 0


    Chip_SCT_EventState(LPC_SCT2, SCT_EVENT_2,  ENABLE_STATE0);			// event 2 only happens in state 0

    Chip_SCT_EventControl(LPC_SCT2, SCT_EVENT_2, (CHIP_SCT_EVENTCTRL_T) ( SCT_IOSEL_RISEIN0	|	// IOSEL   [9:6]   = SCT_IN0
                                                                          SCT_IOCOND_RAISE	|	// IOCOND  [11:10] = rising edge
                                                                          SCT_COMBMODE_IO	  |	// COMBMODE[13:12] = uses IO condition only
                                                                          SCT_STATELD_1		  |	// STATELD [14]    = STATEV is loaded into state
                                                                          SCT_STATEEV_0		  ));	// STATEV  [15]    = new state is 0


    Chip_SCT_CapControl(LPC_SCT2, SCT_CAPCTRL_1, SCT_EVT_1);			// event 1 causes capture 1 to be loaded
    Chip_SCT_CapControl(LPC_SCT2, SCT_CAPCTRL_2, SCT_EVT_2);			// event 2 causes capture 2 to be loaded

    LPC_SCT2->LIMIT_L        = 0x0007;                     				// events 0, 1 and 2 are used as counter limit

    Chip_SCT_EnableEventInt(LPC_SCT2, (CHIP_SCT_EVENT_T) (SCT_EVT_0 | SCT_EVT_2));			// events 0 and 2 generate interrupts

    NVIC_EnableIRQ(SCT2_IRQn);                             				// enable SCT interrupt


    Chip_SCT_ClearControl(LPC_SCT2,SCT_CTRL_HALT_L);					// unhalt it by clearing bit 2 of the CTRL register
}




int main (void) 
{

	SystemCoreClockUpdate();
	Board_Init();

	Chip_Clock_SetSysClockDiv(6);								//SystemCoreClock/6 = 12Mhz (SystemCoreClock = 72Mhz)

	Chip_Clock_EnablePeriphClock(SYSCTL_CLOCK_MUX );			// enable the PMUX clock

	Chip_INMUX_SelectSCT2Src(0, SCT2_INMUX_PIO0_27);			// SCT2_IN0 at P0_27

    USART0_Init(19200);
    RC5_Init();

    PrintString("\f\nLPC15xx SCT-RC5 test January 2014\n\n");

    while (1)
    {
        if (RC5_flag)                              // wait for RC5 code
        {
            RC5_flag = 0;                          // clear flag
            PrintString("RC5 = ");                 // and print it
            PrintByte(RC5_System);
            PrintString(" ");
            PrintByte(RC5_Command);
            PrintString("\n");
        }
        __WFI();
    }
}
